# =====================================================
# INTERNAL PlatformIO Installer
# DO NOT START MANUALLY
# It is called by install_platformio.cmd
# =====================================================

$ErrorActionPreference = "Stop"

# -----------------------------------------------------
# Konfiguration (HIER Version aendern!)
# -----------------------------------------------------
$PythonVersion = "3.14.2"
$PythonDirName = "Python314"

$PythonBaseDir = Join-Path $env:LOCALAPPDATA "Programs\Python\$PythonDirName"
$PythonExe     = Join-Path $PythonBaseDir "python.exe"

$PythonUrl = "https://www.python.org/ftp/python/$PythonVersion/python-$PythonVersion-amd64.exe"


# -----------------------------------------------------
# Git Konfiguration
# -----------------------------------------------------
$GitVersion = "2.45.2"
$GitInstallerName = "Git-$GitVersion-64-bit.exe"
$GitUrl = "https://github.com/git-for-windows/git/releases/download/v$GitVersion.windows.1/$GitInstallerName"

$GitInstallDir = Join-Path $env:ProgramFiles "Git"
$GitExe = Join-Path $GitInstallDir "bin\git.exe"


# -----------------------------------------------------
# Logging (UTF-8, Notepad-kompatibel)
# -----------------------------------------------------
$LogFile = Join-Path $PSScriptRoot "install.log"
if (Test-Path $LogFile) { Remove-Item $LogFile -Force }

function Write-Log {
    param(
        [ValidateSet("INFO","WARN","ERROR")]
        [string]$Level,
        [string]$Message
    )

    $ts = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $line = "[$ts][$Level] $Message"

    Write-Host $line
    [System.IO.File]::AppendAllText(
        $LogFile,
        $line + "`r`n",
        [System.Text.Encoding]::UTF8
    )
}

Write-Log INFO "===== PlatformIO Installationsskript gestartet ====="

# -----------------------------------------------------
# Benutzerhinweis
# -----------------------------------------------------
Write-Host ""
Write-Host "! WICHTIG !"
Write-Host "Waehrend der Installation kann eine Windows-Sicherheitsabfrage erscheinen."
Write-Host "Bitte bestaetigen, falls angezeigt."
Write-Host ""
Start-Sleep -Seconds 3

# -----------------------------------------------------
# Python-Version pruefen (nur Zielpfad)
# -----------------------------------------------------
function Get-PythonVersion {
    param([string]$Exe)

    if (-not (Test-Path $Exe)) { return $null }

    $out = & $Exe --version 2>$null
    if ($out -match "Python (\d+\.\d+\.\d+)") {
        return $matches[1]
    }
    return $null
}

$foundVersion = Get-PythonVersion $PythonExe

if ($foundVersion -eq $PythonVersion) {
    Write-Log INFO "Python $foundVersion bereits korrekt installiert"
}
else {
    if ($foundVersion) {
        Write-Log WARN "Python $foundVersion gefunden, Ziel ist $PythonVersion"
        Write-Log INFO "Bereite gezielten Versionswechsel vor"
    }
    else {
        Write-Log INFO "Python nicht vorhanden"
    }

    # Zielverzeichnis gezielt entfernen (Upgrade / Downgrade)
    if (Test-Path $PythonBaseDir) {
        Write-Log INFO "Entferne bestehendes Python-Verzeichnis: $PythonBaseDir"
        Remove-Item $PythonBaseDir -Recurse -Force
    }

    # Installation
    Write-Log INFO "Installiere Python $PythonVersion"
    $installer = Join-Path $env:TEMP "python-installer.exe"
    Invoke-WebRequest -Uri $PythonUrl -OutFile $installer

    Start-Process $installer `
        "/quiet InstallAllUsers=0 PrependPath=0 Include_test=0" `
        -Wait

    Start-Sleep -Seconds 5

    $foundVersion = Get-PythonVersion $PythonExe
    if ($foundVersion -ne $PythonVersion) {
        Write-Log ERROR "Python Installation fehlgeschlagen ;-("
        exit 1
    }

    Write-Log INFO "Python $foundVersion erfolgreich installiert"
}

# -----------------------------------------------------
# PATH setzen (nur Zielversion)
# -----------------------------------------------------
function Add-ToUserPath {
    param([string]$Dir)

    $current = [Environment]::GetEnvironmentVariable("PATH", "User")
    if ($current -split ';' -contains $Dir) { return }

    [Environment]::SetEnvironmentVariable(
        "PATH",
        "$current;$Dir",
        "User"
    )
    Write-Log INFO "PATH erweitert: $Dir"
}

Add-ToUserPath $PythonBaseDir
Add-ToUserPath (Join-Path $PythonBaseDir "Scripts")

# -----------------------------------------------------
# PlatformIO installieren
# -----------------------------------------------------
try {
    & $PythonExe -c "import platformio" 2>$null
    Write-Log INFO "PlatformIO ist bereits installiert"
}
catch {
    Write-Log INFO "Installiere PlatformIO"
    & $PythonExe -m pip install --user -U platformio --no-warn-script-location
    Write-Log INFO "PlatformIO erfolgreich installiert"
}

# -----------------------------------------------------
# pio.exe PATH
# -----------------------------------------------------
$PioScripts = Join-Path $env:APPDATA "Python\$PythonDirName\Scripts"
Add-ToUserPath $PioScripts

# -----------------------------------------------------
# Git-Version pruefen und installieren
# -----------------------------------------------------
function Get-GitVersion {
    param([string]$Exe)

    if (-not (Test-Path $Exe)) { return $null }

    $out = & $Exe --version 2>$null
    if ($out -match "git version (\d+\.\d+\.\d+)") {
        return $matches[1]
    }
    return $null
}

$foundGitVersion = Get-GitVersion $GitExe

if ($foundGitVersion -eq $GitVersion) {
    Write-Log INFO "Git $foundGitVersion bereits korrekt installiert"
}
else {
    if ($foundGitVersion) {
        Write-Log WARN "Git $foundGitVersion gefunden, Ziel ist $GitVersion"
    }
    else {
        Write-Log INFO "Git nicht vorhanden"
    }

    Write-Log INFO "Installiere Git $GitVersion"
    $gitInstaller = Join-Path $env:TEMP $GitInstallerName
    Invoke-WebRequest -Uri $GitUrl -OutFile $gitInstaller

    Start-Process $gitInstaller `
        "/VERYSILENT /NORESTART /NOCANCEL /SP-" `
        -Wait

    Start-Sleep -Seconds 5

    $foundGitVersion = Get-GitVersion $GitExe
    if ($foundGitVersion -ne $GitVersion) {
        Write-Log ERROR "Git Installation fehlgeschlagen ;-("
        exit 1
    }

    Write-Log INFO "Git $foundGitVersion erfolgreich installiert"
}



# -----------------------------------------------------
# Abschluss
# -----------------------------------------------------
Write-Log INFO "Installation erfolgreich abgeschlossen"

Write-Host ""
Write-Host " PlatformIO Installation abgeschlossen  :-)"
Write-Host " Bitte alle Excel-Fenster schliessen und neu oeffnen"
Write-Host ""

